<?php

/**

 * Makes an API request to the Celtic Tuning endpoint.

 * 

 * @since 1.0

 */

function ct_celtictuning_api_request($endpoint, $body_data = []) {

    $api_key = get_option('ct_celtictuning_settings')['apikey'];

    $body_data['apiKey'] = $api_key;

    

    $response = wp_remote_post($endpoint, array(

        'body' => json_encode($body_data),

        'headers' => array('Content-Type' => 'application/json')

    ));



    $response_code = wp_remote_retrieve_response_code($response);

    

    if ($response_code === 400 || "" == $api_key ) {

        return ['error' => 'Invalid API Key. Please enter a valid API key to proceed.'];

    }



    return json_decode(wp_remote_retrieve_body($response), true);

}



/**

 * Fetches vehicle makes from the Celtic Tuning API.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_makes() {

    return ct_celtictuning_api_request('https://www.celtictuningapi.co.uk/VehicleLookup/GetMakes');

}



/**

 * Fetches vehicle models from the Celtic Tuning API based on make ID.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_models($make_id) {

    return ct_celtictuning_api_request('https://www.celtictuningapi.co.uk/VehicleLookup/GetModels', ['makeId' => $make_id]);

}



/**

 * Fetches fuel types from the Celtic Tuning API based on model ID.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_fuels($model_id) {

    return ct_celtictuning_api_request('https://www.celtictuningapi.co.uk/VehicleLookup/GetFuels', ['modelId' => $model_id]);

}



/**

 * Fetches vehicle variants from the Celtic Tuning API based on model and fuel IDs.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_variants($model_id, $fuel_id) {

    return ct_celtictuning_api_request('https://www.celtictuningapi.co.uk/VehicleLookup/GetVariants', ['modelId' => $model_id, 'fuelId' => $fuel_id]);

}



/**

 * Fetches vehicle details from the Celtic Tuning API based on variant ID.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_vehicles($variantId) {

    return ct_celtictuning_api_request('https://www.celtictuningapi.co.uk/VehicleLookup/GetVehicle', ['variantId' => $variantId]);

}





/**

 * Handles AJAX requests with a callback function.

 * 

 * @since 1.0

 */

function ct_celtictuning_handle_ajax($callback) {

    if (!isset($_POST['make_id']) && !isset($_POST['model_id']) && !isset($_POST['variant_id'])) {

        wp_send_json_error(['message' => 'Missing parameters']);

    }

    $result = call_user_func($callback);

    wp_send_json($result);

}



/**

 * Fetches models via AJAX based on the make ID.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_models_ajax() {

    ct_celtictuning_handle_ajax(function() {

        $make_id = sanitize_text_field($_POST['make_id']);

        return ct_celtictuning_fetch_models($make_id);

    });

}



/**

 * Fetches fuels via AJAX based on the model ID.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_fuels_ajax() {

    ct_celtictuning_handle_ajax(function() {

        $model_id = sanitize_text_field($_POST['model_id']);

        return ct_celtictuning_fetch_fuels($model_id);

    });

}



/**

 * Fetches vehicle variants via AJAX based on model and fuel IDs.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_varients_ajax() {

    ct_celtictuning_handle_ajax(function() {

        $model_id = sanitize_text_field($_POST['model_id']);

        $fuel_id = sanitize_text_field($_POST['fuel_id']);

        return ct_celtictuning_fetch_variants($model_id, $fuel_id);

    });

}



/**

 * Fetches vehicles via AJAX based on the variant ID.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_vehicles_ajax() {

    ct_celtictuning_handle_ajax(function() {

        $variant_id = sanitize_text_field($_POST['variant_id']);

        return ct_celtictuning_fetch_vehicles($variant_id);

    });

}



add_action('wp_ajax_fetch_models', 'ct_celtictuning_fetch_models_ajax');

add_action('wp_ajax_nopriv_fetch_models', 'ct_celtictuning_fetch_models_ajax');



add_action('wp_ajax_fetch_fuels', 'ct_celtictuning_fetch_fuels_ajax');

add_action('wp_ajax_nopriv_fetch_fuels', 'ct_celtictuning_fetch_fuels_ajax');



add_action('wp_ajax_fetch_varients', 'ct_celtictuning_fetch_varients_ajax');

add_action('wp_ajax_nopriv_fetch_varients', 'ct_celtictuning_fetch_varients_ajax');



add_action('wp_ajax_fetch_vehicles', 'ct_celtictuning_fetch_vehicles_ajax');

add_action('wp_ajax_nopriv_fetch_vehicles', 'ct_celtictuning_fetch_vehicles_ajax');



add_action('wp_ajax_vehicle_url', 'vehicle_url');

add_action('wp_ajax_nopriv_vehicle_url', 'vehicle_url');



/**

 * Returns a vehicle URL in an AJAX response.

 * 

 * @since 1.0

 */

function vehicle_url() {

    $variant = $_POST['variant'];

    wp_send_json_success(['message' => 'Data processed successfully.']);

}





/**

 * Fetches full vehicle details including makes, models, and variants.

 * 

 * @since 1.0

 */

function ct_celtictuning_fetch_vehicle_details($variant_id) {

    $vehicle_data = ct_celtictuning_fetch_vehicles($variant_id);



    if (isset($vehicle_data['error'])) {

        return ['error' => $vehicle_data['error']];

    }



    $make_id = $vehicle_data['MakeId'];  

    $model_id = $vehicle_data['ModelId'];

    $fuel_id = $vehicle_data['FuelType'];



    $makes = ct_celtictuning_fetch_makes();

    $models = ct_celtictuning_fetch_models($make_id);

    $fuels = ct_celtictuning_fetch_fuels($model_id);

    $variants = ct_celtictuning_fetch_variants($model_id, $fuel_id);



    return [

        'makes' => $makes,

        'models' => $models,

        'fuels' => $fuels,

        'variants' => $variants,

        'vehicle_data' => $vehicle_data,

    ];

}



add_action('wp_ajax_fetch_vehicle_details', 'fetch_vehicle_details_ajax');

add_action('wp_ajax_nopriv_fetch_vehicle_details', 'fetch_vehicle_details_ajax');



/**

 * Normalizes a model text by converting it to lowercase, replacing periods and non-alphanumeric characters with hyphens.

 *

 * @since 1.0

 */

function normalizeModelText($text) {

    $text = strtolower($text);

    $text = str_replace('.', '-', $text);

    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);

    $text = str_replace(' ', '-', $text);

    $text = preg_replace('/-+/', '-', $text);

    return trim($text);

}



/**

 * Fetches vehicle details via AJAX based on the variant ID.

 * 

 * @since 1.0

 */

function fetch_vehicle_details_ajax() {

    $make_name = isset($_POST['make_name']) ? sanitize_text_field($_POST['make_name']) : '';

    $modal_name = isset($_POST['modal_name']) ? sanitize_text_field($_POST['modal_name']) : '';

    $fuel_name = isset($_POST['fuel_name']) ? sanitize_text_field($_POST['fuel_name']) : '';

    $variant_name = isset($_POST['variant_name']) ? sanitize_text_field($_POST['variant_name']) : '';



    $variant_name_cleaned = str_replace('-ECU-remap-chiptuning', '', $variant_name);

    $normalized_variant_name = normalizeModelText($variant_name_cleaned);



    $makes = ct_celtictuning_fetch_makes();

    $make_value = null;



    foreach ($makes as $make) {

        $normalized_make_text = normalizeModelText($make['Text']);

        if (strcasecmp($normalized_make_text, trim($make_name)) === 0) {

            $make_value = $make['Value'];

            break;

        }

    }



    if ($make_value !== null) {

        $models = ct_celtictuning_fetch_models($make_value);

        if (empty($modal_name)) {

            $vehicle_data = [

                'MakeId' => $make_value,

            ];

            wp_send_json(['make_id' => $make_value, 'makes' => $makes,'models' => $models,'vehicle_data' => $vehicle_data]);

            wp_die();

        }

        $modal_value = null;

        $normalized_modal_name = normalizeModelText($modal_name);



        foreach ($models as $model) {

            $normalized_model_text = normalizeModelText($model['Text']);

            if (strcasecmp($normalized_model_text, $normalized_modal_name) === 0) {

                $modal_value = $model['Value'];

                break;

            }

        }



        if ($modal_value !== null) {

            $fuel_options = ct_celtictuning_fetch_fuels($modal_value);

            $fuel_value = null;

            if (empty($fuel_name)) {

                $vehicle_data = [

                    'MakeId' => $make_value,

                    'ModelId' => $modal_value,

                ];

                wp_send_json(['modal_id' => $modal_value, 'makes' => $makes,'models' => $models,'fuels' => $fuel_options, 'vehicle_data' => $vehicle_data]);

                wp_die();

            }



            foreach ($fuel_options as $fuel) {

                if (strcasecmp(trim($fuel['Text']), trim($fuel_name)) === 0) {

                    $fuel_value = $fuel['Value'];

                    break;

                }

            }



            if ($fuel_value !== null) {

                $variants = ct_celtictuning_fetch_variants($modal_value, $fuel_value);

                $variant_value = null;

                if (empty($normalized_variant_name)) {

                    $vehicle_data = [

                        'MakeId' => $make_value,

                        'ModelId' => $modal_value,

                        'FuelType' => $fuel_value,

                    ];

                    wp_send_json(['modal_id' => $modal_value, 'makes' => $makes,'models' => $models, 'fuels' => $fuel_options, 'variants'=>$variants, 'fuel_type' => $fuel_value, 'vehicle_data' => $vehicle_data]);

                    wp_die();

                }



                foreach ($variants as $variant) {

                    $normalized_variant_text = normalizeModelText($variant['Text']);

                    if (strcasecmp($normalized_variant_text, $normalized_variant_name) === 0) {

                        $variant_value = $variant['Value'];

                        break;

                    }

                }



                if ($variant_value !== null) {

                    $data = ct_celtictuning_fetch_vehicle_details($variant_value);

                    wp_send_json($data);

                } else {

                    wp_send_json(['error' => 'Variant not found.']);

                }

            } else {

                wp_send_json(['error' => 'Fuel type not found.']);

            }

        } else {

            wp_send_json(['error' => 'Modal not found.']);

        }

    } else {

        wp_send_json(['error' => 'Make not found.']);

    }



    wp_die();

}





add_action('wp_ajax_ct_submit_quote', 'ct_submit_quote');

add_action('wp_ajax_nopriv_ct_submit_quote', 'ct_submit_quote');



/**

 * Handles the submission of a vehicle quote request.

 * 

 * @since 1.0

 */

function ct_submit_quote() {



    if (!check_ajax_referer('request_quote_nonce', 'security', false)) {

        wp_send_json_error(['message' => 'Oops! Invalid Details, Please Provide Valid Information.'], 403);

        wp_die(); 

    }



    if (!empty($_POST['first_name'])) {

        wp_send_json_error(['message' => 'Oops! Invalid Details, Please Provide Valid Information.'], 403);

        wp_die();

    }



    $formStartTime = isset($_POST['formStartTime']) ? intval($_POST['formStartTime']) : 0;

    $currentTime = time();



    if ($currentTime - $formStartTime < 5 || $formStartTime == 0) {

        wp_send_json_error(['message' => 'Oops! Invalid Details, Please Provide Valid Information.'], 403);

    }



    $plugin_settings = get_option('ct_celtictuning_settings');



    if($plugin_settings['enable_recaptcha_v3'] == 'true'){



        $recaptcha_secret_key = isset($plugin_settings['recaptcha_secret_key']) ? sanitize_text_field($plugin_settings['recaptcha_secret_key']) : '';

        $recaptcha_token = sanitize_text_field($_POST['recaptcha_token']);



        $response = wp_remote_post('https://www.google.com/recaptcha/api/siteverify', [

            'body' => [

                'secret' => $recaptcha_secret_key,

                'response' => $recaptcha_token,

            ],

        ]);

    

        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        if (is_wp_error($response) || !$response_body['success']) {

            wp_send_json_error(['message' => 'reCAPTCHA verification failed.']);

            wp_die(); 

        }



        $recaptcha_score = isset($response_body['score']) ? floatval($response_body['score']) : 0;

        //$recaptcha_score = 0.4;

        $recaptcha_threshold = 0.5;

        if ($recaptcha_score < $recaptcha_threshold) {

            wp_send_json_error(['message' => 'reCAPTCHA verification failed.']);

            wp_die();

        }

    }



    global $wpdb;

    $table_name = $wpdb->prefix . 'ct_request_quotes';



    $customer_name = sanitize_text_field($_POST['customerName']);

    $varient_id = sanitize_text_field($_POST['varient_id']);

    $tuningStageId = isset($_POST['tuningStageId']) ? sanitize_text_field($_POST['tuningStageId']) : '';

    $email = sanitize_email($_POST['email']);

    $telephone = sanitize_text_field($_POST['telephone']);

    $post_code = sanitize_text_field($_POST['postCode']);

    $address = sanitize_text_field($_POST['address']);

    $city = sanitize_text_field($_POST['city']);

    $state = sanitize_text_field($_POST['state']);

    $country = sanitize_text_field($_POST['country']);

    $additional_comments = sanitize_textarea_field($_POST['additionalComments']);

    $vehicle_year = sanitize_text_field($_POST['vehicleYear']);

    $vehicle_details = stripslashes(sanitize_textarea_field($_POST['vehicle_details']));

    $tuning_stage = sanitize_text_field($_POST['tuningStage']);

    $options = sanitize_textarea_field($_POST['additionalOptions']);

    $alloptions = isset($_POST['alladditionalOptions']) ? json_encode($_POST['alladditionalOptions']) : '[]';

    $decoded_options = json_decode($alloptions, true);

    $consent = isset($_POST['consent']) ? 1 : 0;

    $marketing = isset($_POST['marketing']) ? 1 : 0;

    $accept = isset($_POST['accept']) ? 1 : 0;

    

    $selected_options = isset($_POST['selectedOptions']) ? $_POST['selectedOptions'] : [];

  

    $api_data = [

        'variantId' => $varient_id,

        'tuningStageId' => $tuningStageId,

        'CustomerName' => $customer_name,

        'Email' => $email,

        'Telephone' => $telephone,

        'VehicleYear' => $vehicle_year,

        'Address' => $address,

        'TownCity' => $city,

        'CountyState' => $state,

        'PostCode' => $post_code,

        'Country' => $country,

        'AdditionalComments' => $additional_comments,

    ];



    if ($consent == 1) {

        if (!empty($email)) {

            $api_data['QbaEmail'] = 'true';

        }

        if (!empty($address)) {

            $api_data['QbaPostal'] = 'true';

        }

        if (!empty($telephone)) {

            $api_data['QbaTelephone'] = 'true';

            $api_data['QbaSms'] = 'true';

        }

    }



    if ($marketing == 1) {

        if (!empty($email)) {

            $api_data['LomkEmail'] = 'true';

        }

        if (!empty($address)) {

            $api_data['LomkPostal'] = 'true';

        }

        if (!empty($telephone)) {

            $api_data['LomkTelephone'] = 'true';

            $api_data['LomkSms'] = 'true';

        }

    }



    if (!empty($selected_options)) {

        foreach ($selected_options as $option) {

            $option_key = sanitize_text_field($option['optionKey']);            

            $option_key = preg_replace('/(Available|Avail)$/i', '', $option_key);



            $api_data[$option_key] = $option['isChecked'] ? 'true' : 'false';

        }

    }



    $api_response = ct_celtictuning_api_request('https://www.celtictuningapi.co.uk/VehicleLookup/SubmitWebQuoteRequest', $api_data);

    if (!$api_response['Successful']) {

        if($api_response['DuplicateRequest']){

            wp_send_json_error(['message' => 'The record is already exists.']);

        }else{

            wp_send_json_error(['message' => 'API Validation Failed: ' . implode(', ', $api_response['ValidationErrors'])]);

        }

        return;

    }

    $api_data_json = json_encode($api_data);

    $variant_details = json_decode($vehicle_details, true);

    $from_year = $variant_details['variantFromYear'] ?? 'N/A';

    $to_year = $variant_details['variantToYear'] ?? 'N/A';

    $formatted_year = "$from_year-$to_year";

    

    $wpdb->insert($table_name, array(

        'reference' => $api_response['Reference'],

        'customer_name' => $customer_name,

        'email' => $email,

        'telephone' => $telephone,

        'address' => $address,

        'post_code' => $post_code,

        'city' => $city,

        'state' => $state,

        'country' => $country,

        'additional_comments' => $additional_comments,

        'vehicle_year' => $vehicle_year,

        'variant_id' => $varient_id,

        'varient_details' => $vehicle_details,

        'tuning_stage' => $tuning_stage,

        'options' => $options,

        'consent' => $consent,

        'marketing' => $marketing,

        'accept' => $accept,

        'api_details' => $api_data_json

    ));



    $subject = 'Quotation Request Received';

    $message = "Hello $customer_name,\n\n";

    $message .= "We have successfully received your quotation request with the following vehicle details and it has been assigned the reference number:" . $api_response['Reference']. ".\n\n";



    $message .= "Model Details:\n";

	$message .= "Make: " . $variant_details['makeName'] . "\n";

    $message .= "Model: " . $variant_details['modelName'] . "\n";

    $message .= "Variant: " . $variant_details['variantName'] . "\n";

    $message .= "Vehicle Year: " . $formatted_year . "\n";

    $message .= "Engine CC: " . $variant_details['engineCC'] . "\n";

    $message .= "Fuel Type: " . $variant_details['fuel'] . "\n";

    $message .= "ECU Type: " . $variant_details['ecuType'] . "\n\n";

    

    $message .= "Your Selected Info/Details:\n";

    $message .= "Customer Name: $customer_name\n";

    $message .= "Email: $email\n";

    $message .= "Telephone: $telephone\n";

    $message .= "Address: $address\n";

    $message .= "Post Code: $post_code\n";

    $message .= "City: $city\n";

    $message .= "State: $state\n";

    $message .= "Country: $country\n";

    $message .= "Additional Comments: $additional_comments\n";

    $message .= "Variant: " . $variant_details['variantName'] . "\n";

    $message .= "Vehicle Year: $vehicle_year\n";

    $message .= "Tuning Stage: $tuning_stage\n\n";

    $message .= "Selected options:\n";

    foreach ($decoded_options as $alloptions => $selected) {

        $message .= "$alloptions : " . ($selected) . "\n";

    }

    $message .= "\n\n";

    

    $message .= "Thank you for your enquiry!\n\n";

    $message .= "Best regards,\n";

    $message .= get_option('ct_celtictuning_settings')['business_name'] ?? 'Celtic Tuning Dealer';

  

    $headers = array('Content-Type: text/plain; charset=UTF-8');

    wp_mail($email, $subject, $message, $headers);



    wp_send_json_success('Quote submitted successfully.');

}

