<?php
/**
 * Displays the Quote Requests page in the admin panel, including search, pagination, and delete functionality.
 * 
 * @since 1.0
 */
function ct_view_quote_requests_page() {
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'ct_request_quotes';
    
    if (isset($_GET['delete_all'])) {
        $start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']) : '';
        $end_date = isset($_GET['end_date']) ? sanitize_text_field($_GET['end_date']) : '';
        
        $deleted_rows = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $table_name WHERE timestamp BETWEEN %s AND %s",
                $start_date . ' 00:00:00',
                $end_date . ' 23:59:59'
            )
        );
    
        if ($deleted_rows > 0) {
            set_transient('ct_delete_message', 'Records within the selected date range have been deleted.', 60);
        } else {
            set_transient('ct_delete_message', 'No records were deleted.', 60);
        }
        
        echo '<script type="text/javascript">
            window.location.href = "' . esc_url(admin_url('admin.php?page=ct_view_quote_requests')) . '";
        </script>';
        exit;

    }
    ob_start();

    $delete_message = get_transient('ct_delete_message');
    if ($delete_message) {
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($delete_message) . '</p></div>';
        delete_transient('ct_delete_message'); // Remove the transient after displaying the message
    }

    $search_term = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
    $start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']) : '';
    $end_date = isset($_GET['end_date']) ? sanitize_text_field($_GET['end_date']) : '';
    $page = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
    $items_per_page = 10;
    $offset = ($page - 1) * $items_per_page;
    $sr_no = 1;

    $query = "SELECT * FROM $table_name WHERE customer_name LIKE %s";
    $query_args = array('%' . $wpdb->esc_like($search_term) . '%');

    $query .= " ORDER BY timestamp DESC LIMIT %d OFFSET %d";
    array_push($query_args, $items_per_page, $offset);

    $quote_requests = $wpdb->get_results($wpdb->prepare($query, $query_args));

    $total_count_query = "SELECT COUNT(*) FROM $table_name WHERE customer_name LIKE %s";
    $count_args = array('%' . $wpdb->esc_like($search_term) . '%');

    $total_quotes = $wpdb->get_var($wpdb->prepare($total_count_query, $count_args));
    $total_pages = ceil($total_quotes / $items_per_page);

    echo '<div class="wrap">';
    echo '<h1>Quote Requests</h1>';
    echo '<form method="get" id="ct_request_quote_filter" action="">';
    echo '<input type="hidden" name="page" value="ct_view_quote_requests" />';
    echo '<input type="text" name="s" value="' . esc_attr($search_term) . '" placeholder="Search by name..." />&nbsp;&nbsp;';
    echo '<input type="submit" value="Search" class="button" />&nbsp;&nbsp;';
    echo 'Start Date: <input type="date" id="start_date" name="start_date" class="date-field" value="' . esc_attr($start_date) . '" placeholder="Start Date" />&nbsp;&nbsp;';
    echo 'End Date: <input type="date" id="end_date" name="end_date" class="date-field" value="' . esc_attr($end_date) . '" placeholder="End Date" />&nbsp;&nbsp;';
    echo '<input type="submit" name="delete_all" class="button" value="Delete" onclick="return confirm(\'Are you sure you want to delete these records?\');" />';
    echo '</form>';

    if (!empty($quote_requests)) {
        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead><tr><th>ID</th><th>Name</th><th>Email</th><th>Telephone</th><th>Vehicle Year</th><th>Date</th></tr></thead>';
        echo '<tbody>';
        foreach ($quote_requests as $request) {
            echo '<tr>';
            echo '<td>' . esc_html($sr_no) . '</td>';
            echo '<td>' . esc_html($request->customer_name) . '</td>';
            echo '<td>' . esc_html($request->email) . '</td>';
            echo '<td>' . esc_html($request->telephone) . '</td>';
            echo '<td>' . esc_html($request->vehicle_year) . '</td>';
            echo '<td>' . esc_html($request->timestamp) . '</td>';
            echo '</tr>';
            $sr_no++;
        }
        echo '</tbody></table>';

        $base_url = add_query_arg(array(
            'page' => 'ct_view_quote_requests',
            's' => urlencode($search_term),
            'start_date' => urlencode($start_date),
            'end_date' => urlencode($end_date),
        ), admin_url('admin.php'));

        echo paginate_links(array(
            'base' => $base_url . '%_%',
            'format' => '&paged=%#%',
            'total' => $total_pages,
            'current' => $page,
            'prev_text' => __('« Previous'),
            'next_text' => __('Next »'),
        ));
    } else {
        echo '<p>No quote requests found.</p>';
    }

    echo '</div>';
    echo "<script>
    document.addEventListener('DOMContentLoaded', () => {
        // Target the start and end date inputs
        const startDateInput = document.getElementById('start_date');
        const endDateInput = document.getElementById('end_date');

        const openCalendar = (field) => {
            if (field.showPicker) {
                field.showPicker();
            } else {
                field.focus();
            }
        };

        startDateInput.addEventListener('click', function () {
            openCalendar(this);
        });

        endDateInput.addEventListener('click', function () {
            openCalendar(this);
        });

        startDateInput.addEventListener('change', function () {
            endDateInput.min = this.value;

            if (endDateInput.value && endDateInput.value < this.value) {
                endDateInput.value = ''; // Clear invalid end date
                alert('End date must be later than or equal to the start date.');
            }
        });
    });
</script>";
    ob_end_flush();
}
?>
